<?php
/**
 * Magento
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@magentocommerce.com so we can send you a copy immediately.
 *
 * @category   Unic
 * @package    Unic_Saferpaynew
 * @copyright  Copyright (c) 2004-2008 Unic Internet Solutions (http://www.unic.com)
 * @license    http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

/**
 * Saferpaynew Controller
 *
 * @category   Unic
 * @package    Unic_Saferpaynew
 * @name       Unic_Saferpaynew_SaferpaynewController
 * @author     Unic Internet Solutions <magento@unic.com>
 */

class Unic_SaferpayNew_SaferpayNewController extends Mage_Core_Controller_Front_Action {
    /**
     * Redirect Block Type
     *
     * @var string
     */
    protected $_redirectBlockType = 'saferpaynew/redirect';

    protected function _expireAjax() {

        if (! $this->getCheckout()->getQuote()->hasItems()) {
            $this->getResponse()->setHeader('HTTP/1.1', '403 Session Expired');
            exit();
        }
    }

    /**
     * Get singleton of Checkout Session Model
     *
     * @return Mage_Checkout_Model_Session
     */
    public function getCheckout() {

        return Mage::getSingleton('checkout/session');
    }

    /**
     * this is the action that gets called in function getOrderPlaceRedirectUrl
     */
    public function redirectAction() {

        // Get the current cart
        $session = $this->getCheckout();
        $session->setSaferpaynewQuoteId($session->getQuoteId());
        $session->setSaferpaynewRealOrderId($session->getLastRealOrderId());
        
        // get the order model
        $order = Mage::getModel('sales/order');
        // get the order by its number
        $order->loadByIncrementId($session->getLastRealOrderId());
        // add a message to the history section in order management
        $order->addStatusToHistory(Mage_Sales_Model_Order::STATE_PENDING_PAYMENT, Mage::helper('saferpaynew')->__('Customer was redirected to Saferpaynew.'));
        $order->save();
        
        // a lot of code, but delivers the Block Redirect.php to the users
        $this->getResponse()->setBody($this->getLayout()->createBlock($this->_redirectBlockType)->setOrder($order)->toHtml());
    }

    /**
     * Saferpaynew calls this action if payment was successful
     */
    public function successAction() {

        // check if the response is valid
        $status = $this->_checkReturnedMessage();
        
        if ($status) {
            // get the order information
            $session = $this->getCheckout();
            $session->getQuote()->setIsActive(false)->save();
            
            $order = Mage::getModel('sales/order');
            $order->load($this->getCheckout()->getLastOrderId());
            if ($order->getId()) {
                $order->sendNewOrderEmail();
            }
            
            // everything allright, redirect the user to success page
            $this->_redirect('checkout/onepage/success');
        } else {
            // something is wrong -> redirect to failure controller
            $this->_redirect('saferpaynew/saferpaynew/failure');
        }
    }
    
    /**
     * Saferpaynew calls this action if confirmation page is displayed
     * user independant confirmation to the shop
     */
    public function notifyAction() {
        
        // check if the response is valid
        $status = $this->_checkReturnedMessage(true);
    }

    /**
     * Display cart if saferpay returned to the failure url
     */
    public function failureAction() {

        $session = $this->getCheckout();
        $order = Mage::getModel('sales/order');
        $order->load($this->getCheckout()->getLastOrderId());
        
        $order->addStatusToHistory(Mage_Sales_Model_Order::STATE_CANCELED, Mage::helper('saferpaynew')->__('There was an error at Saferpaynew Payment. Customers payment was not valid.'));
        $order->save();
        
        $this->_redirect('checkout/cart');
    }

    /**
     * Display cart if customer canceled payment
     */
    public function cancelAction() {

        $session = $this->getCheckout();
        $order = Mage::getModel('sales/order');
        $order->load($this->getCheckout()->getLastOrderId());
        
        $order->addStatusToHistory(Mage_Sales_Model_Order::STATE_CANCELED, Mage::helper('saferpaynew')->__('Customer canceled Saferpaynew Payment.'));
        $order->save();
        
        $this->_redirect('checkout/cart');
    }

    /**
     * Get debug flag
     *
     * @return string
     */
    public function getDebug() {

        return Mage::getStoreConfig('payment/saferpaynew/debug_flag');
    }
    
    /**
     * Checking response from saferpay variables.
     * Creating invoice if payment was successfull or cancel order if payment was declined
     */
    protected function _checkReturnedMessage($notify = false) {

        $status = true;
        
        $signature = $_REQUEST['SIGNATURE'];
        $data = $_REQUEST['DATA'];
        $DataRegexp = '/^<IDP\s+([a-zA-Z0-9]+="[a-zA-Z0-9().\s-]*"\s*)*\/>$/';
        $SignatureRegexp = "^[a-zA-Z0-9]+$";
        
        /* Check Signature and Data with a regular expression */
        /*
        if (! (ereg($SignatureRegexp, stripslashes($signature)) &&preg_match($DataRegexp, stripslashes($data)))) {
            $this->_redirect('saferpay/saferpay/failure');
            exit();
        }
        */
        
        // $data contains an xml element, so create an xml object of it
        $dom = new domDocument();
        $dom->loadXML($data);
        $idp = $dom->getElementsByTagName('IDP')->item(0);
        
        $order = Mage::getModel('sales/order');
        // load order by orderid from saferpay response tag
        $order->loadByIncrementId($idp->getAttribute('ORDERID'));
        
        $paymentInst = $order->getPayment()->getMethodInstance();
        
        // add slashes to ensure the console-tool works
        
        $saferpay_payconfirm_gateway = "https://www.saferpay.com/hosting/VerifyPayConfirm.asp";

        // * catch magic_quotes_gpc is set to yes in PHP.ini
        if (substr($data, 0, 15) =="<IDP MSGTYPE=\\\"") {
            $data = stripslashes($data);
        }
        
        // **************************************************
        // * Put all attributes together and create hosting PayConfirm URL 
        // * For hosting: each attribute which could have non-url-conform characters inside should be urlencoded before
        // **************************************************
        
        $payconfirm_url = $saferpay_payconfirm_gateway ."?DATA=" .urlencode($data) ."&SIGNATURE=" .urlencode($signature);
        
        // **************************************************
        // * Get the Payment URL from the saferpay hosting server 
        // **************************************************
        // * Initialize CURL session
        // **************************************************

        $cs = curl_init($payconfirm_url);
        
        // **************************************************
        // * Set CURL-session options
        // **************************************************
        
        curl_setopt($cs, CURLOPT_PORT, 443); // set option for outgoing SSL requests via CURL
        curl_setopt($cs, CURLOPT_SSL_VERIFYPEER, false); // ignore SSL-certificate-check - session still SSL-safe
        curl_setopt($cs, CURLOPT_HEADER, 0); // no header in output
        curl_setopt($cs, CURLOPT_RETURNTRANSFER, true); // receive returned characters
        
        // **************************************************
        // * Execute CURL-session
        // **************************************************

        $verification = curl_exec($cs);
        
        // **************************************************
        // * End CURL-session
        // **************************************************

        curl_close($cs);
        
        // **************************************************
        // * Stop if verification not successful is not working
        // **************************************************

        if (strtoupper(substr($verification, 0, 3)) !="OK:") {
            
            /*
			 * something was wrong with either the command line tool or someone
			 * tried to fake the response, so we redirect to failure page
			 */
            $this->_redirect('saferpaynew/saferpaynew/failure');
            exit();
        }
        
        /*
		 * if everything is alright with the payment, (it should be as saferpay
		 * called the success url, the result from the command line payment confirm
		 * process should be an xml-element as well
		 * so we create a new domDocument
		 */
        $dom_result = new domDocument();
        if ($dom_result->loadXML($data)) {
            $dom_result_idp = $dom_result->getElementsByTagName('IDP')->item(0);
        } else {
            /*
			 * something was wrong with either the command line tool or someone
			 * tried to fake the response, so we redirect to failure page
			 */
            $this->_redirect('saferpaynew/saferpaynew/failure');
            exit();
        }
        
        if ($this->getDebug()) {
            $debugResponse = array();
            $debugResponse['MSGTYPE'] = $idp->getAttribute('MSGTYPE');
            $debugResponse['KEYID'] = $idp->getAttribute('KEYID');
            $debugResponse['ID'] = $idp->getAttribute('ID');
            $debugResponse['AMOUNT'] = $idp->getAttribute('AMOUNT');
            $debugResponse['CURRENCY'] = $idp->getAttribute('CURRENCY');
            $debugResponse['PROVIDERID'] = $idp->getAttribute('PROVIDERID');
            $debugResponse['PROVIDERNAME'] = $idp->getAttribute('PROVIDERNAME');
            $debugResponse['ORDERID'] = $idp->getAttribute('ORDERID');
            $debugResponse['ACCOUNTID'] = $idp->getAttribute('ACCOUNTID');
            $debugResponse['ECI'] = $idp->getAttribute('ECI');
            $debugResponse['CCCOUNTRY'] = $idp->getAttribute('CCCOUNTRY');
            $debugResponse['SIGNATURE'] = $signature;
            
            $tmp = '';
            foreach ($debugResponse as $k => $v) {
                $tmp .= '<' .$k .'>' .$v .'</' .$k .'>';
            }
            
            $debug = Mage::getModel('saferpaynew/api_debug')->setResponseBody($tmp)->setOrderId($debugResponse['ORDERID'])->save();
        }
        
        if ($dom_result_idp->getAttribute('MSGTYPE') =='PayConfirm') {
            /*
			 * now we are sure that the payment was successful as the result xml
			 * element from the command line tool contains an attribute MSGTYPE
			 * with value PayConfirm. We process the payment.
			 */
            
            /*
			 * the following code has been copied from eWay payment module.
			 */
            $invoicable = false;
            if ($order->canInvoice() && $notify) {
                // it's a valid order
                $invoicable = true;
                $convertor = Mage::getModel('sales/convert_order');
                // create the invoice
                $invoice = $convertor->toInvoice($order);
                foreach ($order->getAllItems() as $orderItem) {
                    if (! $orderItem->getQtyToInvoice()) {
                        continue;
                    }
                    $item = $convertor->itemToInvoiceItem($orderItem);
                    $item->setQty($orderItem->getQtyToInvoice());
                    $invoice->addItem($item);
                }
                $invoice->collectTotals();
                $invoice->register()->capture();
                Mage::getModel('core/resource_transaction')->addObject($invoice)->addObject($invoice->getOrder())->save();
                
                $paymentInst->setTransactionId($idp->getAttribute('ID'));
                
                // we add a status message to the message-section in admin
                $order->addStatusToHistory(Mage::getStoreConfig('payment/saferpaynew/order_status'), Mage::helper('saferpaynew')->__('Customer successfully returned from Saferpaynew<br /><strong>Transaction ID: %s</strong>', $idp->getAttribute('ID')));
            }

            if (!$invoicable && $notify) {
                /*
				 * payment was not successfull. this code should not be executed
				 * to make it safe, we redirect to failure action
				 */
                
                $order->cancel();
                $order->addStatusToHistory(Mage_Sales_Model_Order::STATE_CANCELED, Mage::helper('saferpaynew')->__('Customer was rejected by Saferpaynew'));
                $status = false;
                $this->_redirect('saferpaynew/saferpaynew/failure');
            }
            
            $order->save();
        } else {
            /*
			 * payment was not successfull. this code should not be executed
			 * to make it safe, we redirect to failure action
			 */
            
            $order->cancel();
            $order->addStatusToHistory(Mage_Sales_Model_Order::STATE_CANCELED, Mage::helper('saferpaynew')->__('Customer was rejected by Saferpaynew'));
            $status = false;
            $this->_redirect('saferpaynew/saferpaynew/failure');
        }
        return $status;
    }
}